/******************************************************************************
 * (C) Copyright 2000 by Agilent Technologies GmbH. All rights reserved.      *
 ******************************************************************************/


/* ---------------------------------------------------------------
 * File: xsetting.c
 *       functions for setting database
 * -----------------------------------------------------------------*/

#include <xsetting.h>
#include <xaddrmap.h>
#include <xdecode.h>

#ifndef MIN
#define MIN(a,b) ( (a) <= (b) ? (a) : (b) )
#endif

#ifndef MAX
#define MAX(a,b) ( (a) >= (b) ? (a) : (b) )
#endif

/* Pointer to exerciser database */
#define DB_EXERCISER (bx_handlearray[handle].db->Exerciser)

/* Value of generic property */
#define GENERIC(prop) (DB_EXERCISER.GenProp[prop])

static struct 
{ 
  bx_int32 Val;
  bx_int32 Mask; /* 0=Read only, 1=ReadWrite for BIOS */
} 

/* These are the default values and masks for the config space.
   They implicitely contain the following decoder properties:
   BX_DECP_SIZE,BX_DECP_PREFETCH,BX_DECP_LOCATION
*/

DefConfSpaceMephisto[BX_CTCFG_MEMDEPTH]=
{
  /* Configuration header */
  0x292915BC, 0x00000000,    /* 0x00  Device ID, Vendor ID                  */
  0x04100000, 0xF900037F,    /* 0x04  PCI Status, Command Register          */
  0xFF000000, 0x00000000,    /* 0x08  Class Code, Revision ID               */
  0x00004000, 0x0000FFFF,    /* 0x0c  Lat=0x0 instead of 0x40 in PCI-mode!  */
  0x0000000c, 0xFFF00000,    /* 0x10  BAR 0: Mem64,prefetchable,1MB         */
  0x00000000, 0xFFFFFFFF,    /* 0x14  BAR 1: see BAR 0                      */
  0x00000004, 0xFFFF0000,    /* 0x18  BAR 2: Mem64, not prefetchable,64KB   */
  0x00000000, 0xFFFFFFFF,    /* 0x1c  BAR 3: see BAR 2                      */
  0x00000001, 0xFFFFFF00,    /* 0x20  BAR 4: IO,256 Bytes                   */
  0x00000000, 0x00000000,    /* 0x24  BAR 5: unused                         */
  0x00000000, 0x00000000,    /* 0x28  Cardbus CIS Pointer                   */
  0x292915BC, 0x00000000,    /* 0x2c  Subsytem ID, Subsystem Vendor ID      */
  0x00000000, 0x00000000,    /* 0x30  Expansion ROM Base Aaddress           */
  0x00000060, 0x00000000,    /* 0x34  Capability pointer                    */
  0x00000000, 0x00000000,    /* 0x38  Reserved                              */
  0x00000000, 0x00000000,    /* 0x3c  Max_Lat,Min_Gnt,IR-Pin,IR-Line        */

  /* Programming registers: only accessible from PCI bus !!!
     For Mephisto address range,these items are not present !               */
  0x00000000, 0x00000000,    /* 0x40  Programming Data                      */
  0x00000000, 0x00000000,    /* 0x44  Programming Status                    */
  0x00000000, 0x00000000,    /* 0x48  Connection Status and -register       */
  0x00000000, 0x00000000,    /* 0x4c  Mailbox Data                          */
  0x00000000, 0x00000000,    /* 0x50  Card's Status,Intr- and Mailbox Status*/
  0x00000000, 0x00000000,    /* 0x54  Access -address and -command          */
  0x00000000, 0x00000000,    /* 0x58  Access Data                           */
  0x00000000, 0x00000000,    /* 0x5c  Access Status                         */

  /* PCI-X Capability list items 
     For Mephisto address range,these items go from offset 0x40 to 0x5c !   */
  0x00726807, 0x007F0000,    /* 0x60  PCIX-Command Register,Next Cap,Cap ID */
  0x1FE3FFF8, 0x200Cfff8,    /* 0x64  PCIX-Status Register (see (***) below)*/
  0x00027001, 0x00000000,    /* 0x68  Power Management Cap,Next Cap,Cap ID  */
  0x00000000, 0x00000000,    /* 0x6c  Data,Bridge Ext,PM Control/Status Reg */
  0x00800005, 0x00310000,    /* 0x70  Message Control Reg, Next Cap,Cap ID  */
  0x00000000, 0xFFFFFFFC,    /* 0x74  Message Address                       */
  0x00000000, 0xFFFFFFFF,    /* 0x78  Message Upper Address                 */
  0x00000000, 0x0000FFFF     /* 0x7c  Message Data                          */

  /*  (***) The bits 3-15 of the PCIX-Status register's mask are RW
      (in contrast to spec). They contain the device and bus number.
      RW means here that they are updated by each ConfigWrite access 
      adresssing our card WITH THE ADDRESS THE RI PUTS ON THE BUS (see spec).
      This is also true, when the Command register is written to, in 
      which case the DATA bits 0-15 are ignored (thats our spec RO behavior) 
  */
},

DefConfSpaceFaust[BX_CTCFG_MEMDEPTHMAX]=
{
  /* Configuration header */
  0x293015BC, 0x00000000,    /* 0x00  Device ID, Vendor ID                  */
  0x04100000, 0xF900037F,    /* 0x04  PCI Status, Command Register          */
  0xFF000000, 0x00000000,    /* 0x08  Class Code, Revision ID               */
  0x00004000, 0x0000FFFF,    /* 0x0c  Lat=0x0 instead of 0x40 in PCI-mode!  */
  0x0000000c, 0xFFF00000,    /* 0x10  BAR 0: Mem64,prefetchable,1MB         */
  0x00000000, 0xFFFFFFFF,    /* 0x14  BAR 1: see BAR 0                      */
  0x00000004, 0xFFFF0000,    /* 0x18  BAR 2: Mem64, not prefetchable,64KB   */
  0x00000000, 0xFFFFFFFF,    /* 0x1c  BAR 3: see BAR 2                      */
  0x00000001, 0xFFFFFF00,    /* 0x20  BAR 4: IO,256 Bytes                   */
  0x00000000, 0x00000000,    /* 0x24  BAR 5: unused                         */
  0x00000000, 0x00000000,    /* 0x28  Cardbus CIS Pointer                   */
  0x293015BC, 0x00000000,    /* 0x2c  Subsytem ID, Subsystem Vendor ID      */
  0x00000000, 0x00000000,    /* 0x30  Expansion ROM Base Aaddress           */
  0x00000060, 0x00000000,    /* 0x34  Capability pointer                    */
  0x00000000, 0x00000000,    /* 0x38  Reserved                              */
  0x00000000, 0x00000000,    /* 0x3c  Max_Lat,Min_Gnt,IR-Pin,IR-Line        */

  /* Programming registers: only accessible from PCI bus !!!
     For Mephisto address range,these items are not present !               */
  0x00000000, 0x00000000,    /* 0x40  Programming Data                      */
  0x00000000, 0x00000000,    /* 0x44  Programming Status                    */
  0x00000000, 0x00000000,    /* 0x48  Connection Status and -register       */
  0x00000000, 0x00000000,    /* 0x4c  Mailbox Data                          */
  0x00000000, 0x00000000,    /* 0x50  Card's Status,Intr- and Mailbox Status*/
  0x00000000, 0x00000000,    /* 0x54  Access -address and -command          */
  0x00000000, 0x00000000,    /* 0x58  Access Data                           */
  0x00000000, 0x00000000,    /* 0x5c  Access Status                         */

  /* PCI-X Capability list items 
     For Mephisto address range,these items start from offset 0x40 !        */
  0x20727807, 0x007F0000,    /* 0x60  PCIX-Command Register,Next Cap,Cap ID */
  0x1FE3FFF8, 0x200Cfff8,    /* 0x64  PCIX-Status Register (see (***) below)*/
  0x00000000, 0xd000007c,    /* 0x68  ECC Control and Status                */
  0x00000000, 0x00000000,    /* 0x6c  ECC First Address                     */
  0x00000000, 0x00000000,    /* 0x70  ECC Second Address                    */
  0x00000000, 0x00000000,    /* 0x74  ECC Attribute                         */

  /* Power management */
  0x00028001, 0x00000000,    /* 0x78  Power Management Cap,Next Cap,Cap ID  */
  0x00000000, 0x00000000,    /* 0x7c  Data,Bridge Ext,PM Control/Status Reg */
  
  /* Message Control */
  0x00800005, 0x00310000,    /* 0x80  Message Control Reg, Next Cap,Cap ID  */
  0x00000000, 0xFFFFFFFC,    /* 0x84  Message Address                       */
  0x00000000, 0xFFFFFFFF,    /* 0x88  Message Upper Address                 */
  0x00000000, 0x0000FFFF     /* 0x8c  Message Data                          */

  /*  (***) The bits 3-15 of the PCIX-Status register's mask are RW
      (in contrast to spec). They contain the device and bus number.
      RW means here that they are updated by each ConfigWrite access 
      adresssing our card WITH THE ADDRESS THE RI PUTS ON THE BUS (see spec).
      This is also true, when the Command register is written to, in 
      which case the DATA bits 0-15 are ignored (thats our spec RO behavior) 
  */
};


/*---------------------------------------------------------------------------*
 * BestXConfRegDefaultGet
 *
 * Purpose: Returns the default value from config space
 *                 
 *---------------------------------------------------------------------------*/

bx_errtype EXPORT BestXConfRegDefaultGet(
    bx_handletype handle,
    bx_int32 offset,  /* Byte offset, must be DW-aligned */
    bx_int32 * val
)
{
  /* Gets default value of a DW in config space */
  bx_int32 ConfigSize;

  BX_DECLARE_FUNCNAME("BestXConfRegDefaultGet []");

  BX_TRY_VARS_NO_PROG;

  BX_TRY_BEGIN
  {
    ConfigSize=(BestXHasMephisto(handle)?BX_CTCFG_MEMDEPTH:BX_CTCFG_MEMDEPTHMAX);
    BX_TRY_FCT_PARAM(2, offset >= ConfigSize*4);
    BX_TRY_FAIL(offset & 0x03UL ? BX_E_NO_DWORD_BOUNDARY : BX_E_OK);
    BX_TRY_FCT_PARAM_NULL_POINTER(val);
  }
  
  if (BestXHasMephisto(handle))
  {
    *val=DefConfSpaceMephisto[offset/4].Val;  
  }
  else
  {
    *val=DefConfSpaceFaust[offset/4].Val;  
  }

  BX_ERRETURN(BX_TRY_RET);
}

/*---------------------------------------------------------------------------*
 * BestXConfRegMaskDefaultGet
 *
 * Purpose: Returns the default mask from config space
 *                 
 *---------------------------------------------------------------------------*/

bx_errtype EXPORT BestXConfRegMaskDefaultGet(
    bx_handletype handle,
    bx_int32 offset,
    bx_int32 * val
)
{
  /* Gets mask of a DW in config space */
  bx_int32 ConfigSize;

  BX_DECLARE_FUNCNAME("BestXConfRegMaskDefaultGet []");

  BX_TRY_VARS_NO_PROG;

  BX_TRY_BEGIN
  {
    ConfigSize=(BestXHasMephisto(handle)?BX_CTCFG_MEMDEPTH:BX_CTCFG_MEMDEPTHMAX);
    BX_TRY_FCT_PARAM(2, offset >= ConfigSize*4);
    BX_TRY_FAIL(offset & 0x03UL ? BX_E_NO_DWORD_BOUNDARY : BX_E_OK);
    BX_TRY_FCT_PARAM_NULL_POINTER(val);
  }
  
  if (BestXHasMephisto(handle))
  {
    *val=DefConfSpaceMephisto[offset/4].Mask;  
  }
  else
  {
    *val=DefConfSpaceFaust[offset/4].Mask;  
  }

  BX_ERRETURN(BX_TRY_RET);
}

/*---------------------------------------------------------------------------*
 * BestXDBInit
 *
 * Purpose: Inits and defaults the whole database of the handle array.
 *
 *          For each pointer in DB:
 *            If DB already exists: 
 *              Free memory and set pointer to NULL 
 *            else (e.g. call from BestOpen()): 
 *              Set pointer to NULL (ignoring current pointer value)
 *
 *---------------------------------------------------------------------------*/

bx_errtype EXPORT BestXDBInit(bx_handletype handle)
{
  bx_int32 i,j,k;

  BX_DECLARE_FUNCNAME("BestXDBInit []");

  BX_TRY_VARS_NO_PROG;

  BX_TRY_BEGIN
  {
    /* We always need CAPI license (chris) */
    BX_TRY_LICENSE(BX_CAPABILITY_CAPI); 
  
    if (bx_handlearray[handle].db)
    {
      /* There is already a database, so we first have to free the old one */

      /* Free the pattern terms */
      for (i=0;i<BX_PATT_ENUMSIZE;i++)
      {
        /* We may free NULL here !! */
        BestXMemFree((void**)(bx_handlearray[handle].db->Pattern+i)); 
      }

      /* Free the sequencer condition properties */
      for (i=0;i<BX_TRIGSEQ_SIZE;i++) /* for all trigger-sequencers */
      {
        for (j=0;j<BX_TRIGLINE_SIZE;j++) /* for all lines */
        {
          for (k=0;k<BX_TRIGLINECOND_SIZE;k++) /* for all condition props */
          {
            /* We may free NULL here !! */
            BestXMemFree((void**)(bx_handlearray[handle].db->Analyzer.TrigSeq[i].Line[j].Condition+k)); 
          }
        }
      }
      for (i=0;i<BX_PERFSEQ_SIZE;i++) /* for all performance-sequencers */
      {
        for (j=0;j<BX_PERFLINE_SIZE;j++) /* for all lines */
        {
          for (k=0;k<BX_PERFLINECOND_SIZE;k++) /* for all condition props */
          {
            /* We may free NULL here !! */
            BestXMemFree((void**)(bx_handlearray[handle].db->Analyzer.PerfSeq[i].Line[j].Condition+k)); 
          }
        }
      }

      /* Free the whole database */
      BestXMemFree((void**)&(bx_handlearray[handle].db));
    }
    else
    {
      /* We first go here, because all members of bx_handlearray[handle]
         are initialized to NULL in xsession.c (static array initialization)
      */
    }

    /* Allocate a new DB */
    bx_handlearray[handle].db=(bx_dbtype*) BestXMemMalloc (sizeof(bx_dbtype));
    if (bx_handlearray[handle].db==NULL)
    {
      BX_ERRETURN(BX_E_HOST_MEM_FULL);
    }

    /* Init pattern strings */
    for (i=0;i<BX_PATT_ENUMSIZE;i++)
    {
      bx_handlearray[handle].db->Pattern[i]=(char *)NULL;
    }

    /* Trigger sequencer condition props */
    for (i=0;i<BX_TRIGSEQ_SIZE;i++) /* for all trigger-sequencers */
    {
      for (j=0;j<BX_TRIGLINE_SIZE;j++) /* for all lines */
      {
        for (k=0;k<BX_TRIGLINECOND_SIZE;k++) /* for all condition props */
        {
          /* We may free NULL here !! */
          bx_handlearray[handle].db->Analyzer.TrigSeq[i].Line[j].Condition[k]=NULL; 
        }
      }
    }

    /* Performance sequencer condition props */
    for (i=0;i<BX_PERFSEQ_SIZE;i++) /* for all performance-sequencers */
    {
      for (j=0;j<BX_PERFLINE_SIZE;j++) /* for all lines */
      {
        for (k=0;k<BX_PERFLINECOND_SIZE;k++) /* for all condition props */
        {
          /* We may free NULL here !! */
          bx_handlearray[handle].db->Analyzer.PerfSeq[i].Line[j].Condition[k]=NULL; 
        }
      }
    }


    /***************************************************************************/      
    /*** Set defaults to avoid writing nonsense to card ************************/
    /***************************************************************************/      

    BX_TRY(BestXAllDefaultSet(handle));
    
  } /* TRY */

  BX_ERRETURN(BX_TRY_RET);
}

/*---------------------------------------------------------------------------*
 * BestXDBClose
 *
 * Purpose: Frees the whole database of the handle array
 *---------------------------------------------------------------------------*/

bx_errtype BestXDBClose(bx_handletype handle)
{
  BX_DECLARE_FUNCNAME("BestXDBClose []");

  bx_int32 i,j,k;

  if (bx_handlearray[handle].db)
  {
    /* There is a database */

    /* Free the pattern strings */
    for (i=0;i<BX_PATT_ENUMSIZE;i++)
    {
      BestXMemFree((void**)(bx_handlearray[handle].db->Pattern+i)); 
    }

    /* Free the sequencer condition properties */
    for (i=0;i<BX_TRIGSEQ_SIZE;i++) /* for all trigger-sequencers */
    {
      for (j=0;j<BX_TRIGLINE_SIZE;j++) /* for all lines */
      {
        for (k=0;k<BX_TRIGLINECOND_SIZE;k++) /* for all condition props */
        {
          /* We may free NULL here !! */
          BestXMemFree((void**)(bx_handlearray[handle].db->Analyzer.TrigSeq[i].Line[j].Condition+k)); 
        }
      }
    }
    for (i=0;i<BX_PERFSEQ_SIZE;i++) /* for all performance-sequencers */
    {
      for (j=0;j<BX_PERFLINE_SIZE;j++) /* for all lines */
      {
        for (k=0;k<BX_PERFLINECOND_SIZE;k++) /* for all condition props */
        {
          /* We may free NULL here !! */
          BestXMemFree((void**)(bx_handlearray[handle].db->Analyzer.PerfSeq[i].Line[j].Condition+k)); 
        }
      }
    }

    /* Free the whole database */
    BestXMemFree((void**)&(bx_handlearray[handle].db));
  }

  BX_ERRETURN(BX_E_OK);
}

/*---------------------------------------------------------------------------*
 * BestXAllDefaultSet
 *
 * Purpose: Defaults all values in DB.
 * Precondition: An allocated and initialized DB must already be present.
 *---------------------------------------------------------------------------*/
bx_errtype EXPORT BestXAllDefaultSet (bx_handletype handle)
{
  BX_DECLARE_FUNCNAME("BestXAllDefaultSet [alldefset]");

  BX_TRY_VARS_NO_PROG;

  bx_int32 i;

  BX_TRY_BEGIN
  {
    /* Generic properties */
    BX_TRY(BestXAnalyzerGenDefaultSet(handle));

    /* Observer properties */
    BX_TRY(BestXObsDefaultSet(handle));

    /* Trigger sequencer */
    BX_TRY(BestXTrigGenDefaultSet(handle));
    BX_TRY(BestXTrigDefaultSet(handle));

    /* Performance sequencer */
    if(BestXHasFirmware(handle))
    {
      for (i=0;i<BX_PERFSEQ_SIZE;i++) 
      {
        BX_TRY(BestXPerfDefaultSet(handle,i));
        BX_TRY(BestXPerfGenDefaultSet(handle,i));
      }
    }
    /* Board props */
    BX_TRY(BestXBoardDefaultSet(handle));

    /* Default exerciser properties; mark memories on card to be valid */
    BX_TRY(BestXExerciserDefaultSet(handle));  

    /* Default pattern terms */
    BX_TRY(BestXPattDefaultSet(handle));  

  } /* TRY */ 

  BX_ERRETURN(BX_TRY_RET);
}


void EXPORT SetUpdateInfo(
  bx_memupdtype *Update,   /* Pointer to update info struct in DB  */
  bx_bool  Touched,        /* Wether to reset or update the struct */
  bx_int32 Row,            /* memory Row (line), which has been changed */
  bx_int32 Col)            /* memory Col (property), which has been changed */
{
  /* This functions sets info for updating 
     an exerciser memory (behavior,block).
     Should be called instead of updating the struct directly.
   */
  
  if (Update==NULL)
  {
    /* This is not considered to be an error */
    return;
  }

  Update->Touched=Touched;
  
  if (Touched)
  {
    /* Memory has changed, maybe because of a previous 
       call to a PropertySet() function.
       We have to update the Row- and Col members 
     */   
     
    if (Update->MinRow>Update->MaxRow||Update->MinCol>Update->MaxCol) /* reset values */
    {
      /* first memory change */
      Update->MinRow=Update->MaxRow=Row;
      Update->MinCol=Update->MaxCol=Col;
    }
    else
    {
      /* Already valid entries.
         Update only if range becomes larger
       */
      Update->MinRow=MIN(Row,Update->MinRow);
      Update->MaxRow=MAX(Row,Update->MaxRow);
      Update->MinCol=MIN(Col,Update->MinCol);
      Update->MaxCol=MAX(Col,Update->MaxCol);
    }
  }
  else 
  {
    /* This happens after a card update.
       HW- and host memory are identical.
       Set to invalid value combinations here,
       indicating memory is up-to-date.
     */
    Update->MinRow=1;
    Update->MaxRow=0;
    Update->MinCol=1;
    Update->MaxCol=0;
  }    
}

void EXPORT GetUpdateInfo(
  bx_memupdtype * Update,
  bx_bool  *Touched,
  bx_int32 *FirstRow,
  bx_int32 *LastRow,
  bx_int32 *FirstCol,
  bx_int32 *LastCol
)
{
  /* This functions gets info for updating 
     an exerciser memory (behavior,block).
     Should be called instead of reading the struct directly.
   */

  if (Update==NULL)
  {
    /* This is not considered to be an error */
    return;
  }

  if (Touched)  *Touched =Update->Touched;
  if (FirstRow) *FirstRow=Update->MinRow;
  if (LastRow)  *LastRow =Update->MaxRow;
  if (FirstCol) *FirstCol=Update->MinCol;
  if (LastCol)  *LastCol =Update->MaxCol;
}  

